#ifndef _WISE_SDK_ECR_H_
#define _WISE_SDK_ECR_H_

#include "error_code_ecr.h"

#ifdef _WIN32
#include <WINDOWS.H>
#endif

#ifdef __cplusplus
extern "C"
{
#endif


// The connection type
#define CONNECTION_TYPE_USB     1       // Create the connection via USB interface
#define CONNECTION_TYPE_LAN     2       // Create the connection via LAN (Wi-Fi or Ethernet)

#pragma pack(4)
typedef struct
{        
    char    szTerminalMacAddr[18];      // The MAC address of the terminal, ending with ‘\0’. for example, "00:1A:2B:3C:4D:5E".
    
    char    szTerminalName[65];         // The terminal name, ending with ‘\0’.
    BYTE    bTerminalNameLen;           // The length of the terminal name which includes the ‘\0’ terminator.

    char    szAliasName[65];            // The terminal alias name, ending with ‘\0’.
    BYTE    bAliasNameLen;              // The length of the terminal alias name which includes the ‘\0’ terminator.
    
    char    szTerminalIp[40];           // The IP address of the terminal, ending with ‘\0’. for example, "192.168.1.2".
    BYTE    bTerminalIpLen;             // The length of IP address which includes the ‘\0’ terminator.

    DWORD   dwPort;                     // The IP port
    
    char    szTerminalSn[33];           // The terminal serial number, ending with ‘\0’
    BYTE    bTerminalSnLen;             // The length of the terminal serial number which includes the ‘\0’ terminator.
}ST_TERMINAL_INFO;
#pragma pack()

typedef struct
{
	//This callback function is called when Terminal requests a pairing
    void (__stdcall *onRequestPairing)(IN ST_TERMINAL_INFO *pstTerminalInfo);

	//This callback function is called when Terminal requests to be unpaired
    void (__stdcall *onRequestUnpairing)(IN ST_TERMINAL_INFO *pstTerminalInfo);   
}ST_ECR_LAN_PAIRING_CALLBACK;

typedef struct
{
	//This callback function is called when a network connection is established
    void (__stdcall *onConnected)(void);

	//This callback function is called when the network connection is disconnected
    void (__stdcall *onDisconnected)(void);

	//This callback function is called when an error occurs during connection establishment
    void (__stdcall *onError)(IN DWORD dwErrCode, IN char *pszErrMsg);   
}ST_ECR_CONNECTION_CALLBACK;

typedef struct
{        
    void (__stdcall *onSuccess)(IN char *pszResponse);

    void (__stdcall *onError)(IN DWORD dwErrCode, IN char *pszErrMsg);
}ST_ECR_TRANS_CALLBACK;


/**
 * @brief SDK initialization.
 * 
 * This function is used to initialize the SDK.This function performs the necessary initialization internally. 
 *	The application must call this function first before calling any other function, but it only needs to be called once.
 * 
 * @param dwConnectionType This parameter specifies the communication interface between the ECR Register and the Terminal, 
 *	which can be CONNECTION_TYPE_USB and CONNECTION_TYPE_LAN.
 * @return The return code of the function.
 */
DWORD __stdcall ECR_Init(IN DWORD dwConnectionType); 


/**
 * @brief Waiting for pairing request.
 * 
 * This function is used to establish a network pairing between ECR Register and Terminal.
 *	Inside this function, an mDNS broadcast is made and a websocket listening service is started for receiving pairing 
 *	requests from Terminal.
 * 
 * @param pstPairingCallback This is the callback function used to handle the pairing process with Terminal.
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_WaitPairing(IN ST_ECR_LAN_PAIRING_CALLBACK *pstPairingCallback);


/**
 * @brief Stop pairing.
 * 
 * This function is used to stop the pairing process that has been started.
 * 
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_StopWaitPairing(void);


/**
 * @brief Accept paring.
 * 
 * This function is used to accept the pairing request sent by Terminal.
 * After accepting the pairing request from the Terminal, this function will permanently save the information of 
 *	the successfully paired Terminal internally. Unless the application calls the ECRLAN_DeletePairList function 
 *	to delete the pairing information, the pairing information will always exist.
 * 
 * @param pstTerminalInfo Terminal information.
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_AcceptPairing(IN ST_TERMINAL_INFO *pstTerminalInfo);  


/**
 * @brief Reject pairing.
 * 
 * This function is used to reject the pairing request sent by Terminal.
 * 
 * @param pstTerminalInfo Terminal信息。
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_RejectPairing(IN ST_TERMINAL_INFO *pstTerminalInfo);   


/**
 * @brief Unpairing.
 * 
 * This function will notify the Terminal and terminate the pairing relationship with the Terminal. However, the current 
 *	network connection will not be disconnected. In addtion, this function will automatically delete the Terminal from the 
 *	internal pairing list.
 * 
 * @param pstTerminalInfo Terminal information.
 * @return The return code of the function.
 */
void __stdcall ECRLAN_Unpair(IN ST_TERMINAL_INFO *pstTerminalInfo);   

/**
 * @brief Get pairing information.
 * 
 * This function is used to get a list of all the pairing information between and Terminal maintained within this SDK.
 * 
 * @param pstTerminalInfo A pointer to an array of structures used to hold information about terminals that have ever been 
 *	successfully paired. When this parameter is NULL, this function should return ERROR_SUCCESS and the pdwNum parameter 
 *	should return the actual number of paired terminals.
 * @param pdwNum As an input parameter, it indicates the maximum expected number of paired terminals to be obtained.
 *	If the size of pstTerminalInfo is insufficient, ERR_ECR_BUFFER_NOT_ENOUGH should be returned.
 *	As an output parameter, it indicates the number of terminals actually returned that were ever paired successfully. 
 * 
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_GetPairList(OUT ST_TERMINAL_INFO *pstTerminalInfo, IN OUT DWORD *pdwNum); 

/**
 * @brief Delete the pairing list.
 * 
 * This function is used to remove all terminals or a specified Terminal from the list of pairing information. this function 
 *	does not need to notify the Terminal
 * 
 * @param pstTerminalInfo Information for the specified terminal. To delete all paired terminals, set to NULL.
 * @param dwFlag A flag. If the value is 0, the specified terminal is deleted; if the value is 1, all paired terminals are deleted.
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_DeletePairList(IN ST_TERMINAL_INFO *pstTerminalInfo, IN DWORD dwFlag);

/**
 * @brief Establish network connection.
 * 
 * This function is used to establish a network connection with the Terminal.
 * 
 * @param pstTerminalInfo Terminal information.
 * @param dwWaitingSeconds Timeout time in seconds.
 * @param pstConnectionCallback Callback function for the connection.
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_Connect(IN ST_TERMINAL_INFO *pstTerminalInfo, IN DWORD dwWaitingSeconds, 
						IN ST_ECR_CONNECTION_CALLBACK *pstConnectionCallback); 

/**
 * @brief Disconnect network connection.
 * 
 * This function is used to disconnect the network connection with the current Terminal.
 * 
 * @return The return code of the function.
 */
DWORD __stdcall ECRLAN_Disconnect(void); 

/**
 * @brief Check network connection status.
 * 
 * This function is used to check the status of the network connection with Terminal.
 * 
 * @return Network connection status. 
 *	TRUE: the network is connected; 
 *	FALSE: the network is in the disconnected state.
 */
BOOL __stdcall ECRLAN_isConnected(void);


/**
 * @brief Establish USB connection.
 * 
 * This function is used to establish a USB connection with Terminal.
 * 
 * @param pstConnectionCallback Callback function for the connection.
 * @return The return code of the function.
 */
DWORD __stdcall ECRUSB_Connect(IN ST_ECR_CONNECTION_CALLBACK *pstConnectionCallback);   


/**
 * @brief Disconnect USB connection.
 * 
 * This function is used to disconnect the USB connection with Terminal.
 * 
 * @return The return code of the function.
 */
DWORD __stdcall ECRUSB_Disconnect(void); 


/**
 * @brief Check USB connection status.
 * 
 * This function is used to check the USB connection status between Register and Terminal.
 * 
 * @return The return code of the function.
 *	TRUE: Connected; 
 *	FALSE: USB is disconnected.
 */
BOOL __stdcall ECRUSB_isConnected(void);


/**
 * @brief Perform transaction.
 * 
 * This function is used to call Terminal to perform a transaction process, including Purchase, Cashback, Void, pre-authorization and so on.
 * 
 * @param pszRequestMessage Information about the transaction message sent by the ECR Register to the Terminal. The message is in Json data format.
 * @param dwWaitingSeconds The timeout time for the transaction in seconds. If Terminal does not return at the set time, WAIT_TIMEOUT is returned.
 * @param pstTransCallback Callback function. In the correct case, onSuccess is called, and the pszResponse parameter returns the Terminal's 
 *	response message (in Json format). In case of error, onError is called.
 * @return The return code of the function.
 */
DWORD __stdcall ECR_DoTransaction(IN char *pszRequestMessage, IN DWORD dwWaitingSeconds, IN ST_ECR_TRANS_CALLBACK *pstTransCallback);

/**
 * @brief Cancel transaction.
 * 
 * This function is used to cancel the Terminal's execution of a transaction. Note that not all transactions can be canceled, only some 
 *	of the transactions that are in the waiting for user interaction state can be canceled.
 * 
 * @param pszRequestMessage Message data about the transaction sent by the ECR Register to the Terminal. The message is in Json format.
 * @return The return code of the function.
 */
DWORD __stdcall ECR_CancelTransaction(IN char *pszRequestMessage, IN DWORD dwWaitingSeconds);


/**
 * @brief Query transaction records. 
 * 
 * This function is used to query Terminal for transaction records. 
 * 
 * @param pszRequestMessage Message data sent by the ECR Register to the Terminal. The message is in Json format.
 * @param dwWaitingSeconds The timeout time for the transaction in seconds. If Terminal does not return at the set time, WAIT_TIMEOUT is returned.
 * @param pstTransCallback Callback function. In the correct case, onSuccess is called, and the pszResponse parameter returns the Terminal's 
 *	response message (in Json format). In case of error, onError is called.
 * @return The return code of the function.
 */
DWORD __stdcall ECR_QueryTransaction(IN char *pszRequestMessage, IN DWORD dwWaitingSeconds, IN ST_ECR_TRANS_CALLBACK *pstTransCallback);


#ifdef __cplusplus
}
#endif

#endif //_WISE_SDK_ECR_H_


